from django import forms
from django.urls import reverse, path
from django.utils.html import format_html
from django.contrib import admin
from .models import Transaction, StockHistory
from App.templatetags.filters import rupiah
from django.db.models import F, Sum
from django.shortcuts import render
from django.utils.safestring import mark_safe
from unfold.admin import ModelAdmin
from unfold.contrib.import_export.forms import ExportForm, ImportForm, SelectableFieldsExportForm
from django.utils.translation import gettext_lazy as _
# Register your models here.

@admin.register(Transaction)
class TransactionAdmin(ModelAdmin, admin.ModelAdmin):
    list_display = ['transaction_date', 'employee_name', 'customer_name', 'status', 'formatted_total_amount', 'formatted_paid', 'formatted_change', 'print_invoice', ]    
    search_fields = ['employee__name', 'customer__name', 'status']
    list_per_page = 10
    
    def has_add_permission(self, request):
        # Return False to disable the add button
        return False
    
    def has_change_permission(self, request, obj=None):
        # Jika user hanya memiliki view_permission, maka tidak bisa mengubah data
        if request.user.has_perm('transaction.view_transaction') and not request.user.has_perm('transaction.change_transaction'):
            return False
        return True
    
    def transaction_date(self, obj):
        if self.has_change_permission(self.request):
            # Jika user punya permission change, tampilkan sebagai link
            return format_html('<a href="{}">{}</a>', 
                reverse('admin:transaction_transaction_change', args=[obj.pk]), 
                obj.transaction_date)
        # Jika user tidak punya permission change, tampilkan hanya text
        return obj.transaction_date
    
    def employee_name(self, obj):
        return obj.employee.name  # Ambil nama dari relasi ForeignKey
    employee_name.short_description = 'Employee'  # Label kolom di admin

    def customer_name(self, obj):
        return obj.customer.name  # Ambil nama dari relasi ForeignKey
    customer_name.short_description = 'Customer'  # Label kolom di admin

    def print_invoice(self, obj):
        url = reverse('print_invoice', args=[obj.id])  # URL untuk cetak invoice
        return format_html('<a class="button" href="{}" target="_blank">Invoice</a>', url)

    print_invoice.short_description = 'Print Invoice'
    print_invoice.allow_tags = True

    # Metode untuk memformat harga menggunakan filter
    def formatted_total_amount(self, obj):
        return format_html('<div style="text-align: right;">{}</div>', rupiah(obj.total_amount))  # Memformat price dengan rupiah filter
    formatted_total_amount.short_description = "Total"

    def formatted_paid(self, obj):
        return format_html('<div style="text-align: right;">{}</div>', rupiah(obj.paid_amount))
    formatted_paid.short_description = "Paid"

    def formatted_change(self, obj):
        return format_html('<div style="text-align: right;">{}</div>', rupiah(obj.change_amount))
    formatted_change.short_description = "Change"